/* ------------------------------------------------------------------------------
  File: chr6d_usart.h
  Author: CH Robotics
  Version: 1.0
  
  Description: Function declarations for USART communication
------------------------------------------------------------------------------ */ 
#include "stm32f10x.h"

#ifndef __CHR_USART_H
#define __CHR_USART_H

// Size of TX and RX buffers. These buffers hold raw transmitted and received data from the USART.
// These buffers should never have to be accessed directly by user code.
#define	RX_BUF_SIZE					50
#define 	TX_BUF_SIZE     			50

// Maximum number of data bytes that can be stored in a packet
#define	MAX_PACKET_DATA			40

// Sizes of buffers for storing RX and TX packet data.  The TX Packet Buffer is used
// to store packets that are "waiting" to be transmitted.  This is necessary since 
// multiple resources may need access to the USART hardware simultaneously.  The TX buffer
// acts as a moderator, storing packets and transmitting them when the hardware becomes available.
// The RX Packet Buffer stores packets received over the USART receiver.  Since multiple packets
// may arrive before a packet is processed, the RX Packet Buffer is neccessary to ensure that no data
// is lost.  Both the TX and RX Packet Buffers are circular.
#define	TX_PACKET_BUFFER_SIZE	10
#define	RX_PACKET_BUFFER_SIZE	3

// Definitions of states for USART receiver state machine (for receiving packets)
#define	USART_STATE_WAIT			1
#define	USART_STATE_TYPE			2
#define	USART_STATE_ADDRESS		3
#define	USART_STATE_DATA			4
#define	USART_STATE_CHECKSUM		5

// Flags for interpreting the packet type byte in communication packets
#define	PACKET_HAS_DATA			(1 << 7)
#define	PACKET_IS_BATCH			(1 << 6)
#define	PACKET_BATCH_LENGTH_MASK	(0x0F)

#define	PACKET_BATCH_LENGTH_OFFSET	2

#define	BATCH_SIZE_2				2
#define	BATCH_SIZE_3				3

#define	PACKET_NO_DATA				0
#define	PACKET_COMMAND_FAILED	(1 << 0)

// Define flags for identifying the type of packet address received
#define	ADDRESS_TYPE_CONFIG		0
#define	ADDRESS_TYPE_DATA			1
#define	ADDRESS_TYPE_COMMAND		2

// Union for working with floats at the byte level
typedef union __fconvert 
{
	 uint32_t uint32_val;
	 int32_t int32_val;
	 float float_val;
} fConvert;

// Buffer, buffer index, and TX status flag for USART transmit
extern volatile char gTXBuf[TX_BUF_SIZE];
extern volatile int32_t gTXBufPtr;
extern volatile char gTXBusy;
extern uint8_t gUSART_State;

// USART RX buffer and associated index and flags
extern volatile char gRXBuf[RX_BUF_SIZE];
extern volatile int32_t gRXBufPtr;
extern volatile char gRXPacketReceived;
extern volatile char gRXBufOverrun;
	 
// Structure for storing TX and RX packet data
typedef struct __USARTPacket
{
	 uint8_t PT;				// Packet type
	 uint8_t address;			// Packet address
	 uint16_t checksum;		// Checksum
	 
	 // Data included for convenience, but that isn't stored in the packet itself
	 uint8_t data_length; 	// Number of bytes in data section 
	 uint8_t address_type;	// Specified the address type (DATA, CONFIG, OR COMMAND)
	 
	 uint8_t packet_data[MAX_PACKET_DATA];
} USARTPacket;

// USART interface functions
int32_t USART1_transmit( char* txdata, int32_t length );
int32_t TXBufPush( char txdata );
char TXBufPop( void );
void USART1_TX_start( void );

void hexPrint16( uint16_t byte );
void hexPrint8( uint8_t byte );

void USART1_IRQHandler(void);
void DMA1_Channel4_IRQHandler( void );

// Function for copying next packet in the TXPacketBuffer into the TX buffer.
void SendNextPacket( void );
void AddTXPacket( USARTPacket* new_packet );
void AddRXPacket( USARTPacket* new_packet );

void HandleUSART1Reception( void );
void ProcessNextCharacter( void );

uint16_t ComputeChecksum( USARTPacket* new_packet );

void SendTXPacket( USARTPacket* new_packet );
void SendTXPacketSafe( USARTPacket* new_packet );

// Global queue for packets to be transmitted when the UART transmitter becomes available.
extern volatile USARTPacket gTXPacketBuffer[TX_PACKET_BUFFER_SIZE];
extern volatile uint8_t gTXPacketBufferStart;
extern volatile uint8_t gTXPacketBufferEnd;

extern volatile char TXPacketBufferReady;
extern volatile char gCopyingTXPacketToBuffer;

// Global queue for storing received packets
extern volatile USARTPacket gRXPacketBuffer[RX_PACKET_BUFFER_SIZE];
extern volatile uint8_t gRXPacketBufferStart;
extern volatile uint8_t gRXPacketBufferEnd;

#endif